//	Transfer Function.cpp 	
//	written by Dr. Sam Green, W0PCE 
//	for the Fully Automated DDS Sweep Generator Measurement System 
//
//	This program sweeps the D/A converter from a lower to a higher output voltage
//	with a number of steps.   Set parameters in the command line.  
//	Defaults are -2.5V to +2.5V and 4095 steps.  
//	The A/D measures voltage at input 1 at each step and records the data to a text file.  
//	You must specify a text file name as the first paramter following the command 
//  TRANSFER.EXE OUTPUTFILENAME.TXT [START] [END] [NUMBEROFSTEPS] 
//	with START and END in integer millivolts between -2500 and +2500 
//	and an integer NUMBEROFSTEPS
/*
	This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
//  Driver for MAXIM MX7543 serial D to A Converter  
//  Set up for bipolar operation,
//  0x000 gives (2048/2048) x (-Vref)
//  0x800 gives 0 volts
//  0xfff gives (2047/2048) x (+Vref)
//  With Vref set by LM385-25 to 2.5 volts, range is from -2.5V to +2.499V 
//		or a resolution of 1.22 mV 

//  Driver for MAXIM DAC110 serial A to D Converter with calibration procedure  
//  With +/-Vref set by LM385-1.2s to +/-1.234V volts, range is from -2.468V to +2.468V 
//		for a resolution of about 300 uV when running at 13 bit resolution   
//  timeout to prevent lock-up 

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <windows.h>

void write12databits2c0 (int dacword) ;	// DAC 
int writereaddatabits (int writeword) ;	// ADC
void cal (void);						// ADC 
void gotoxy (int x, int y) ;

float tempdata,
	floatreading;

int base	= 0x378,		// parallel port register addresses
	status	= base+1,
	control	= base+2,
	// D/A converter
	dacstart = 0x000,		// default -Vref*2047/2048 start
	dacend   = 0xfff, 		// default +Vref*2047/2048 end
	numberofsteps = 0xfff,	// max 0xfff = 4095 	
	dacword = dacstart,
	stepcount = (dacend-dacstart)/numberofsteps, 
	// A/D converter
	writeword1	= 0x8100,	// input 1 and divide clock by 4 - replace five places to change input!!
	writeword2	= 0x8110,	// input 2 and divide clock by 4
	CONVbits	= 0x0600,	// default 13 bits and 50 Hz 
	CSactive	= 0xbf,	    // active low so set D6 = 0 
	CSinactive	= 0xff,	    // all ones
	ClockLow	= 0xdf,	    // set D5 = 0 
	DataBitLow	= 0xef,	    // set D4 = 0 
	DataBitHi	= 0xff,	    // all ones 
	value, tt=0, intreading, sign, overflow, cc ;

const double 
	mult = 42.94967296,		// = 0x100000000/100000000 
	// A/D converter
	ADC14cal = 2.46670,		// conversion constants vary (~ sum of reference diode voltages) 
	ADC13cal = 2.46955, 
	ADC12cal = 2.46390,  
	// D/A converter
	Vref = 2.5919,			// for LM385-25 
	offset = 0.010 ;		// software correction of offset instead of trimpot 

int 	VrefmV = Vref * 1000;

double  duration ;

clock_t start, now ,then, current, finish;
FILE *fp; 

int main(int argc, char *argv[]) {
	float dacfloat;
	system("CLS");
	dacword = dacstart;  
	if (argc < 2){  fprintf(stderr,"Usage: %s filename.txt [Vstart] [Vend] [#steps]\n\n", argv[0]);
					puts("Vstart less than Vend and both between -2470 to +2470 mV\n");
					puts("in integer mV with no decimal or units\n"); exit(1);}

	if ((fp = fopen(argv[1],"w")) == NULL) {fprintf(stderr,"Can't open %s\n", argv[1]); exit(2);}
	fprintf(fp,"Vsource\tVmeasure\n");

	if (argc>2) {dacstart = atoi(argv[2]);	// enter dacword start in decimal mV 
	if (dacstart < -1*VrefmV) {dacstart = -1*VrefmV; printf("minimum Vsource is %d mV!\n",-1*VrefmV);}
	if (dacstart >    VrefmV) {dacstart =    VrefmV; printf("maximum Vsource is %d mV!\n",   VrefmV);}
		dacfloat=float(dacstart);
		dacfloat = 1 - offset + dacfloat/(Vref*1000); 
		dacstart = dacfloat * 2048;}
	if (argc>3) {dacend   = atoi(argv[3]);	// enter dacword end in decimal mV
	if (dacend < -1*VrefmV)   {dacend   = -1*VrefmV; printf("minimum Vsource is %d mV!\n",-1*VrefmV);}
	if (dacend >    VrefmV)   {dacend   =    VrefmV; printf("maximum Vsource is %d mV!\n",   VrefmV);}
		dacfloat=float(dacend);
		dacfloat = 1 - offset + dacfloat/(Vref*1000); 
		dacend = dacfloat * 2048;} 
	if (dacend>0xfff) exit(1); if (dacstart<0) exit(1);
	if (argc>4) numberofsteps = atoi(argv[4]); 
	stepcount = (dacend-dacstart)/numberofsteps; 
	if (stepcount == 0) stepcount =1; 

	gotoxy(3,2);printf("\tStart = %d\tEnd = %d\n",dacstart,dacend); 
	gotoxy(3,4);printf("\tStart \t\t\t= %1.3fV  ", Vref*(-1 + offset + float(dacstart)/2048)); 
	gotoxy(3,5);printf("\tEnd \t\t\t= %1.3fV    ", Vref*(-1 + offset + float(dacend)  /2048));
	gotoxy(3,6);printf("\tNumber of Steps\t\t= %d   ",numberofsteps);
	gotoxy(3,7);printf("\tCount Increment\t\t= %d    \n", stepcount);	

	cal() ;						
	_outp(control, 8);_outp(control, 0);// pulse  pin 17 high & low (optional o'scope trigger)  
	for(dacword = dacstart; dacword < dacend; dacword += stepcount) {
		write12databits2c0 (dacword) ;

					// measure ADC and write to file 		
		value = writereaddatabits (writeword1 | CONVbits);	// returns previously requested measurement 
		value = writereaddatabits (writeword1 | CONVbits);	// so read twice to get current measurement 
		sign = value & 0x8000 ;
		overflow = value & 0x4000 ;
		intreading = 0xffffffff & value ; 
		if (sign == 0) intreading &= 0x00003fff; else intreading |= 0xffffc000; 
		floatreading = float (intreading * ADC13cal) / 0x3fff ; 
		gotoxy(3,9);printf("Vsource\t= %+1.3f\t Vmeasure\t=  %+1.3f    ", 
			tempdata=Vref*(-1 + offset + float(dacword)/2048), floatreading) ;  
		fprintf(fp,"%1.3f\t%1.3f", tempdata, floatreading) ;
		if (sign != overflow <<1) fprintf(fp,"\tOverflow\n"); else fprintf(fp,"\t\t        \n");
		fflush(fp);
} return 0;} 

void write12databits2c0 (int dacword) { // D/A converter routine 
	int dacvar,i,writebit,mask=0x1000;	// with clock to c1 & transfer update strobe to c2 

	dacvar = ~dacword ;					// bitwise invert to write data to inverted C0 
	for(i=1; i<13; i++) {           
		mask >>= 1 ; writebit = dacvar & mask ; if (writebit!=0) writebit = 1; // reverses dacvar 
		_outp(base+2, 4 + writebit);	// write bit & keep both load (4) and clear (8) inactive
		_outp(base+2, 6 + writebit);	// writeclock low thru inverted C1 & keep both load and clear inactive
		_outp(base+2, 4 + writebit);	// writeclock high thru inverted C1 & keep both load and clear inactive
		}
	_outp(base+2, 0);					//  update stobe pulse low noninverted C2
	_outp(base+2, 4);					//  update stobe pulse high  
}

void cal (void) {											// 3 A/D converter calibration procedure steps 
	int step1 = 0xc, step2 = 8, step3 = 4, value, tt; 
	
	value = writereaddatabits (writeword1 | CONVbits | step1);  
	while((tt=0x80&_inp(base+1)) != 0) ;	 
	value = writereaddatabits (writeword1 | CONVbits | step2);  
	while((tt=0x80&_inp(base+1)) != 0) ; 
	value = writereaddatabits (writeword1 | CONVbits | step3);  
	while((tt=0x80&_inp(base+1)) != 0) ;  
}

int writereaddatabits (int writeword) {						// A/D converter measurement routine 
	int i, temp, writebyte, mask=0x10000, databit, dataword=0;

	_outp(base, CSactive & DataBitLow);			// pull CS low by writing a 0 to D6
	for(i=0; i<16; i++) {           
		writebyte = CSactive & ClockLow & DataBitLow; 
		_outp(base, writebyte);					// write clock low
		mask >>= 1 ; temp = writeword & mask ;	// setup data to write to Din - comes out MSB first
		if (temp==0) temp = DataBitHi; else temp = DataBitLow; 
		_outp(base, writebyte |= ~temp);
		_outp(base, writebyte|= ~ClockLow);		// write clock high 
		databit = 0x40 & _inp(base + 1) ;		// read data bit 6 from DOUT 
		if (databit != 0) {dataword |= mask;}	// build dataword one bit at a time from MSB to LSB 
		}	
	_outp(base, CSinactive & DataBitLow);		// pull CS hi 
	start = clock();
	while((tt=0x80 & _inp(base+1)) != 0) {if(clock() - start > 220) break;} // in milliseconds
	return dataword; 
}

void gotoxy (int x, int y){						// position CRT cursor
	HANDLE hdl;
	COORD coords;
	hdl = GetStdHandle(STD_OUTPUT_HANDLE);
	coords.X=x-1;
	coords.Y=y-1;
	SetConsoleCursorPosition(hdl,coords);
}
